// Configuration
const INTERFERE_CONFIG = {
  mode: 'ratio',                                    // 'always' | 'ratio' | 'domainWhitelist'
  ratio: 0.8,                                      // ratio 模式下拦截概率
  strategy: 'layered',                             // 'random' | 'layered' | 'forced'
  weights: { layered: 0.5, forced: 0.5 },         // random策略下的权重
  layered: { 
    layers: 3,                                     // 层数（2-4层）
    countdownSecOnLayer2: 3                        // 第2层倒计时秒数
  },
  forced: { 
    hoverSeconds: 3,                               // 悬停所需秒数
    area: 'body'                                   // 'body' | 'panel' - 悬停区域
  },
  whitelist: [],                                   // 域名白名单
  maxInterferencePerPage: 5                        // 本页最多干预次数
};

const OVERLAY_Z_INDEX = 2147483646;

// 新增文案池（英文）
const INTERFERENCE_TEXTS = [
  'What were you about to click just now?',
  'Is this link important?',
  'Redirecting your attention...',
  'This is not an advertisement, this is an intentional interruption.',
  'Please maintain your gaze, button unlocking soon.',
  'The next click might go smoothly, or it might not.',
  'Your choice is being delayed.',
  'While you wait, structure is at work.'
];

let shadowHost = null;
let shadowRootRef = null;
let pendingNav = null; // { url, opener: 'same-tab' | 'new-tab' }
let scrollLockedCount = 0;
let currentStrategy = null; // 当前使用的策略
let hasShownInterference = false; // 标记是否已经显示过弹窗

function lockScroll() {
  scrollLockedCount += 1;
  if (scrollLockedCount === 1) {
    document.documentElement.dataset.__dilOriginalOverflow = document.documentElement.style.overflow || '';
    document.body.dataset.__dilOriginalOverflow = document.body.style.overflow || '';
    document.documentElement.style.overflow = 'hidden';
    document.body.style.overflow = 'hidden';
  }
}

function unlockScroll() {
  scrollLockedCount = Math.max(0, scrollLockedCount - 1);
  if (scrollLockedCount === 0) {
    document.documentElement.style.overflow = document.documentElement.dataset.__dilOriginalOverflow || '';
    document.body.style.overflow = document.body.dataset.__dilOriginalOverflow || '';
  }
}

// SessionStorage 管理
function getInterferenceCount() {
  try {
    const count = sessionStorage.getItem('__dil_interference_count__');
    return count ? parseInt(count, 10) : 0;
  } catch { return 0; }
}

function incrementInterferenceCount() {
  try {
    const count = getInterferenceCount() + 1;
    sessionStorage.setItem('__dil_interference_count__', String(count));
    return count;
  } catch { return 0; }
}

function isDomainWhitelisted() {
  try {
    const hostname = window.location.hostname;
    return INTERFERE_CONFIG.whitelist.some(domain => hostname.includes(domain));
  } catch { return false; }
}

function shouldIntercept() {
  // 如果已经显示过弹窗，不再拦截
  if (hasShownInterference) {
    return false;
  }

  // 检查上限（暂时移除限制，确保总是拦截）
  // if (getInterferenceCount() >= INTERFERE_CONFIG.maxInterferencePerPage) {
  //   return false;
  // }

  // 检查白名单
  if (INTERFERE_CONFIG.mode === 'domainWhitelist' && !isDomainWhitelisted()) {
    return false;
  }

  // 总是拦截（不再随机）
  return true;
}

async function ensureShadowRoot() {
  if (shadowRootRef) return shadowRootRef;
  shadowHost = document.createElement('div');
  shadowHost.id = '__dil_shadow_host__';
  shadowHost.style.all = 'initial';
  shadowHost.style.position = 'fixed';
  shadowHost.style.inset = '0';
  shadowHost.style.zIndex = String(OVERLAY_Z_INDEX);
  shadowHost.style.pointerEvents = 'none';
  document.documentElement.appendChild(shadowHost);
  shadowRootRef = shadowHost.attachShadow({ mode: 'open' });
  try {
    const cssUrl = chrome.runtime.getURL('styles.css');
    const resp = await fetch(cssUrl);
    const text = await resp.text();
    const styleEl = document.createElement('style');
    styleEl.textContent = text;
    shadowRootRef.appendChild(styleEl);
  } catch (e) {
    const styleEl = document.createElement('style');
    styleEl.textContent = `
      .dil-overlay{position:fixed;inset:0;background:rgba(0,0,0,0.55);display:flex;align-items:center;justify-content:center}
      .dil-block{background:#fff;border-radius:12px;box-shadow:0 20px 60px rgba(0,0,0,.25);overflow:hidden}
      .dil-btn{background:#111;color:#fff;border:none;border-radius:8px;padding:10px 14px;cursor:pointer}
      .dil-row{display:flex;gap:8px;align-items:center}
    `;
    shadowRootRef.appendChild(styleEl);
  }
  return shadowRootRef;
}

function removeOverlay() {
  const root = shadowRootRef;
  if (!root) return;
  const overlay = root.querySelector('.dil-overlay');
  if (overlay) overlay.remove();
  if (shadowHost) shadowHost.style.pointerEvents = 'none';
  unlockScroll();
  // 标记已经显示过弹窗，后续不再拦截
  hasShownInterference = true;
}

function randomSize() {
  const vw = Math.max(320, Math.min(window.innerWidth, 1440));
  const vh = Math.max(320, Math.min(window.innerHeight, 900));
  const w = Math.round(vw * (0.28 + Math.random() * 0.22));
  const h = Math.round(vh * (0.28 + Math.random() * 0.22));
  return { w, h };
}

function randomChoice(arr) { 
  return arr[Math.floor(Math.random() * arr.length)]; 
}

function buildButton(label, variant = 'primary', disabled = false) { 
  const btn = document.createElement('button'); 
  btn.className = `dil-btn dil-${variant}`; 
  btn.type = 'button'; 
  btn.textContent = label;
  btn.disabled = disabled;
  if (disabled) {
    btn.classList.add('dil-disabled');
  }
  return btn; 
}

function mountOverlay(contentBuilder) {
  const root = shadowRootRef; 
  if (!root) return;
  removeOverlay(); 
  lockScroll();
  
  const overlay = document.createElement('div');
  overlay.className = 'dil-overlay';
  overlay.setAttribute('role', 'dialog');
  overlay.setAttribute('aria-modal', 'true');
  
  const { w, h } = randomSize();
  const panel = document.createElement('div');
  panel.className = 'dil-block';
  panel.style.width = `${w}px`; 
  panel.style.maxWidth = '95vw';
  panel.style.height = `${h}px`; 
  panel.style.maxHeight = '95vh';
  panel.style.display = 'flex'; 
  panel.style.flexDirection = 'column'; 
  panel.style.position = 'relative';

  // 删除关闭按钮
  // const topbar = document.createElement('div'); 
  // topbar.className = 'dil-topbar';
  // const closeBtn = buildButton('Close', 'closer'); 
  // closeBtn.setAttribute('aria-label', 'Close');
  // closeBtn.addEventListener('click', removeOverlay);
  // topbar.appendChild(closeBtn); 
  // panel.appendChild(topbar);

  contentBuilder(panel);

  overlay.appendChild(panel); 
  root.appendChild(overlay);
  
  if (shadowHost) shadowHost.style.pointerEvents = 'auto';
  
  setTimeout(() => { 
    if (shadowHost && shadowRootRef && shadowRootRef.querySelector('.dil-overlay')) { 
      shadowHost.style.pointerEvents = 'auto'; 
    } 
  }, 0);
  
  // 删除 ESC 键关闭功能
  // const escHandler = (ev) => { 
  //   if (ev.key === 'Escape') { 
  //     ev.preventDefault(); 
  //     removeOverlay(); 
  //   } 
  // }; 
  // window.addEventListener('keydown', escHandler, { once: true });
  
  return { overlay, panel };
}

// 多层干扰 - 状态机实现
function showLayeredInterference(onContinue) {
  const layers = Math.max(2, Math.min(4, INTERFERE_CONFIG.layered.layers));
  let currentLayer = 1;
  
  function showLayer(layerNum) {
    const { panel } = mountOverlay((p) => {
      const header = document.createElement('div');
      header.className = 'dil-header';
      header.setAttribute('role', 'heading');
      header.setAttribute('aria-level', '2');
      
      const body = document.createElement('div');
      body.className = 'dil-body';
      
      const footer = document.createElement('div');
      footer.className = 'dil-footer';
      
      if (layerNum === 1) {
        // Layer A
        header.textContent = 'Are you sure you want to visit this page?';
        const msg = document.createElement('div');
        msg.className = 'dil-msg';
        msg.textContent = randomChoice(INTERFERENCE_TEXTS);
        body.appendChild(msg);
        
        const continueBtn = buildButton('Continue', 'primary');
        const cancelBtn = buildButton('Cancel', 'secondary');
        continueBtn.addEventListener('click', () => {
          if (layers > 1) {
            showLayer(2);
          } else {
            onContinue();
          }
        });
        cancelBtn.addEventListener('click', removeOverlay);
        footer.append(continueBtn, cancelBtn);
        
      } else if (layerNum === 2 && layers >= 2) {
        // Layer B - 含倒计时
        header.textContent = 'System is confirming your intention...';
        const msg = document.createElement('div');
        msg.className = 'dil-msg';
        msg.textContent = randomChoice(INTERFERENCE_TEXTS);
        body.appendChild(msg);
        
        const countdownEl = document.createElement('div');
        countdownEl.className = 'dil-countdown';
        countdownEl.setAttribute('role', 'status');
        countdownEl.setAttribute('aria-live', 'polite');
        let count = INTERFERE_CONFIG.layered.countdownSecOnLayer2;
        countdownEl.textContent = `Please wait ${count} second${count !== 1 ? 's' : ''}...`;
        body.appendChild(countdownEl);
        
        const continueBtn = buildButton('Continue', 'primary', true);
        const cancelBtn = buildButton('Cancel', 'secondary');
        
        const timer = setInterval(() => {
          count -= 1;
          if (count <= 0) {
            clearInterval(timer);
            countdownEl.textContent = 'Ready. You can continue now.';
            continueBtn.disabled = false;
            continueBtn.classList.remove('dil-disabled');
            continueBtn.focus();
          } else {
            countdownEl.textContent = `Please wait ${count} second${count !== 1 ? 's' : ''}...`;
          }
        }, 1000);
        
        continueBtn.addEventListener('click', () => {
          if (!continueBtn.disabled) {
            clearInterval(timer);
            if (layers > 2) {
              showLayer(3);
            } else {
              onContinue();
            }
          }
        });
        cancelBtn.addEventListener('click', () => {
          clearInterval(timer);
          removeOverlay();
        });
        footer.append(continueBtn, cancelBtn);
        
      } else if (layerNum === 3 && layers >= 3) {
        // Layer C (可能最后一层)
        header.textContent = 'Final confirmation, are you really leaving?';
        const msg = document.createElement('div');
        msg.className = 'dil-msg';
        msg.textContent = randomChoice(INTERFERENCE_TEXTS);
        const submsg = document.createElement('div');
        submsg.className = 'dil-msg';
        submsg.style.marginTop = '8px';
        submsg.style.fontSize = '0.9em';
        submsg.textContent = 'After leaving, your attention will be redistributed.';
        body.append(msg, submsg);
        
        const continueBtn = buildButton('Continue to original link', 'primary');
        const stayBtn = buildButton('Stay on this page', 'secondary');
        
        if (layers > 3) {
          // 还有第4层
          continueBtn.addEventListener('click', () => showLayer(4));
        } else {
          continueBtn.addEventListener('click', onContinue);
        }
        stayBtn.addEventListener('click', removeOverlay);
        footer.append(continueBtn, stayBtn);
        
      } else if (layerNum === 4 && layers >= 4) {
        // Layer D (最后一层)
        header.textContent = 'Last step: Do you really want to proceed?';
        const msg = document.createElement('div');
        msg.className = 'dil-msg';
        msg.textContent = randomChoice(INTERFERENCE_TEXTS);
        body.appendChild(msg);
        
        const continueBtn = buildButton('Continue to original link', 'primary');
        const stayBtn = buildButton('Stay on this page', 'secondary');
        continueBtn.addEventListener('click', onContinue);
        stayBtn.addEventListener('click', removeOverlay);
        footer.append(continueBtn, stayBtn);
      }
      
      p.append(header, body, footer);
    });
  }
  
  showLayer(1);
}

// 强制注视模式
function showForcedAttention(onContinue) {
  const hoverSeconds = INTERFERE_CONFIG.forced.hoverSeconds || 3;
  const hoverArea = INTERFERE_CONFIG.forced.area || 'body';
  let hoverProgress = 0;
  let hoverTimer = null;
  let isHovering = false;
  let isCompleted = false; // 标记是否已完成
  
  const { overlay, panel } = mountOverlay((p) => {
    const header = document.createElement('div');
    header.className = 'dil-header';
    header.setAttribute('role', 'heading');
    header.setAttribute('aria-level', '2');
    header.textContent = randomChoice(INTERFERENCE_TEXTS);
    
    const body = document.createElement('div');
    body.className = 'dil-body';
    body.setAttribute('tabindex', '0');
    
    const msg = document.createElement('div');
    msg.className = 'dil-msg';
    msg.textContent = 'Please maintain your gaze on the highlighted area to unlock.';
    body.appendChild(msg);
    
    // 进度条容器
    const progressContainer = document.createElement('div');
    progressContainer.className = 'dil-progress-container';
    
    const progressBar = document.createElement('div');
    progressBar.className = 'dil-progress-bar';
    progressBar.setAttribute('role', 'progressbar');
    progressBar.setAttribute('aria-valuemin', '0');
    progressBar.setAttribute('aria-valuemax', String(hoverSeconds));
    progressBar.setAttribute('aria-valuenow', '0');
    
    const progressFill = document.createElement('div');
    progressFill.className = 'dil-progress-fill';
    
    const progressText = document.createElement('div');
    progressText.className = 'dil-progress-text';
    progressText.textContent = `0 / ${hoverSeconds} seconds`;
    progressText.setAttribute('aria-live', 'polite');
    
    progressBar.appendChild(progressFill);
    progressContainer.append(progressBar, progressText);
    body.appendChild(progressContainer);
    
    const footer = document.createElement('div');
    footer.className = 'dil-footer';
    
    const continueBtn = buildButton('Continue to original link', 'primary', true);
    const cancelBtn = buildButton('Cancel', 'secondary');
    
    // 悬停目标区域
    const hoverTarget = hoverArea === 'panel' ? p : body;
    hoverTarget.classList.add('dil-hover-target');
    
    // 鼠标离开事件处理函数（需要单独定义以便后续移除）
    const mouseLeaveHandler = () => {
      if (!isCompleted) {
        stopHover();
      }
    };
    
    function updateProgress(elapsed) {
      hoverProgress = Math.min(hoverSeconds, elapsed);
      const percentage = (hoverProgress / hoverSeconds) * 100;
      progressFill.style.width = `${percentage}%`;
      progressBar.setAttribute('aria-valuenow', String(hoverProgress));
      progressText.textContent = `${hoverProgress.toFixed(1)} / ${hoverSeconds} seconds`;
      
      if (hoverProgress >= hoverSeconds && !isCompleted) {
        isCompleted = true;
        // 停止定时器
        if (hoverTimer) {
          clearInterval(hoverTimer);
          hoverTimer = null;
        }
        // 移除鼠标离开事件监听，防止重置
        hoverTarget.removeEventListener('mouseleave', mouseLeaveHandler);
        // 启用按钮
        continueBtn.disabled = false;
        continueBtn.classList.remove('dil-disabled');
        progressText.textContent = 'Ready! You can continue now.';
        continueBtn.focus();
      }
    }
    
    function startHover() {
      if (isHovering || isCompleted) return;
      isHovering = true;
      const startTime = Date.now() - (hoverProgress * 1000);
      
      hoverTimer = setInterval(() => {
        const elapsed = (Date.now() - startTime) / 1000;
        updateProgress(elapsed);
      }, 50);
    }
    
    function stopHover() {
      if (isCompleted) return; // 如果已完成，不重置
      isHovering = false;
      if (hoverTimer) {
        clearInterval(hoverTimer);
        hoverTimer = null;
      }
      hoverProgress = 0;
      progressFill.style.width = '0%';
      progressBar.setAttribute('aria-valuenow', '0');
      progressText.textContent = `0 / ${hoverSeconds} seconds`;
      continueBtn.disabled = true;
      continueBtn.classList.add('dil-disabled');
    }
    
    // 鼠标事件
    hoverTarget.addEventListener('mouseenter', startHover);
    hoverTarget.addEventListener('mouseleave', mouseLeaveHandler);
    
    // 触屏设备支持
    let touchStartTime = null;
    hoverTarget.addEventListener('touchstart', (e) => {
      if (isCompleted) return;
      e.preventDefault();
      touchStartTime = Date.now();
      startHover();
    });
    
    hoverTarget.addEventListener('touchend', () => {
      if (touchStartTime && !isCompleted) {
        stopHover();
        touchStartTime = null;
      }
    });
    
    hoverTarget.addEventListener('touchcancel', () => {
      if (touchStartTime && !isCompleted) {
        stopHover();
        touchStartTime = null;
      }
    });
    
    continueBtn.addEventListener('click', () => {
      if (!continueBtn.disabled) {
        stopHover();
        onContinue();
      }
    });
    cancelBtn.addEventListener('click', () => {
      stopHover();
      removeOverlay();
    });
    
    footer.append(continueBtn, cancelBtn);
    p.append(header, body, footer);
  });
}

// 选择策略
function selectStrategy() {
  const strategy = INTERFERE_CONFIG.strategy;
  
  if (strategy === 'layered') {
    return 'layered';
  } else if (strategy === 'forced') {
    return 'forced';
  } else {
    // random 模式
    const rand = Math.random();
    const layeredWeight = INTERFERE_CONFIG.weights?.layered || 0.5;
    return rand < layeredWeight ? 'layered' : 'forced';
  }
}

// 主拦截处理函数
function proceedNavigation() { 
  if (!pendingNav) { 
    removeOverlay(); 
    return; 
  } 
  const { url, opener } = pendingNav; 
  window.postMessage({ source: 'dil-content', type: 'PROCEED_NAV', url, opener }, '*'); 
  pendingNav = null; 
  removeOverlay(); 
}

function handleIntercept(url, opener) { 
  console.log('[DIL] handleIntercept called:', url);
  if (!shouldIntercept()) {
    console.log('[DIL] shouldIntercept returned false');
    return false;
  }
  
  // 增加计数
  incrementInterferenceCount();
  console.log('[DIL] Interference count:', getInterferenceCount());
  
  pendingNav = { url, opener }; 
  ensureShadowRoot().then(() => {
    currentStrategy = selectStrategy();
    console.log('[DIL] Selected strategy:', currentStrategy);
    
    if (currentStrategy === 'layered') {
      showLayeredInterference(() => proceedNavigation());
    } else if (currentStrategy === 'forced') {
      showForcedAttention(() => proceedNavigation());
    }
  }); 
  return true; 
}

// 豁免检测
function isExemptLink(anchor, eventPath){
  try {
    const keywords=[
      'ad','ads','advert','advertise','advertising','promo','promotion','promoted','sponsor','sponsored','recommend','recommended','reco','suggest','suggested','related',
      'ai','gpt','copilot','assistant','summary','summarize','智能','广告','推广','赞助','推荐','为你推荐','相关'
    ];
    const attrHit=(el)=>{
      const attrs=['rel','aria-label','aria-labelledby','role','data-section','data-module','data-area','data-testid','data-ad','data-ads','data-sponsored','data-recommend'];
      for (const a of attrs){ if(el.hasAttribute && el.hasAttribute(a)){ const v=String(el.getAttribute(a)||'').toLowerCase(); if(keywords.some(k=>v.includes(k))) return true; } }
      return false;
    };
    const textHit=(el)=>{ try{ const t=(el.innerText||'').slice(0,64).toLowerCase(); return keywords.some(k=>t.includes(k)); }catch{return false;} };
    const classIdHit=(el)=>{ try{ const c=(el.className||'').toString().toLowerCase(); const i=(el.id||'').toString().toLowerCase(); return keywords.some(k=>c.includes(k)||i.includes(k)); }catch{return false;} };
    const path = eventPath && eventPath.length ? eventPath : (anchor.composedPath?anchor.composedPath():[]);
    const nodes = path && path.length ? path : (function(){ const arr=[]; let n=anchor; for(let i=0;i<8&&n;i++){ arr.push(n); n=n.parentElement; } return arr; })();
    for (let idx=0; idx<Math.min(12,nodes.length); idx++){ const el = nodes[idx]; if(!el || el===window || el===document) continue; if (attrHit(el) || classIdHit(el) || textHit(el)) return true; }
    if (anchor.rel && anchor.rel.toLowerCase().includes('sponsored')) return true;
    return false;
  } catch { return false; }
}

function isAnchorOnlyNavigation(anchor) { 
  try { 
    if (!anchor) return false; 
    const href = anchor.getAttribute('href') || ''; 
    if (href.startsWith('#')) return true; 
    const aUrl = new URL(anchor.href); 
    const curr = new URL(window.location.href); 
    return aUrl.origin === curr.origin && aUrl.pathname === curr.pathname && aUrl.search === curr.search && aUrl.hash !== curr.hash; 
  } catch { return false; } 
}

function handleClickEvent(e) {
  try {
    const path = e.composedPath ? e.composedPath() : (e.path || []);
    let anchor = null; 
    for (const el of path) { 
      if (el && el.tagName === 'A') { 
        anchor = el; 
        break; 
      } 
    }
    if (!anchor) {
      console.log('[DIL] No anchor found');
      return;
    }
    // 移除豁免检查，确保100%拦截
    // if (isExemptLink(anchor, path)) {
    //   console.log('[DIL] Link is exempt');
    //   return;
    // }
    const href = anchor.getAttribute('href'); 
    // 只排除真正的无效链接（javascript: 和 download）
    if (!href || href.startsWith('javascript:') || anchor.hasAttribute('download')) {
      console.log('[DIL] Invalid href:', href);
      return;
    }
    // 移除锚点导航检查，确保100%拦截
    // if (isAnchorOnlyNavigation(anchor)) {
    //   console.log('[DIL] Anchor-only navigation');
    //   return;
    // }
    const intendedNewTab = e.button === 1 || e.metaKey === true || e.ctrlKey === true || anchor.target === '_blank';
    const url = anchor.href; 
    console.log('[DIL] Attempting to intercept:', url);
    const intercepted = handleIntercept(url, intendedNewTab ? 'new-tab' : 'same-tab'); 
    console.log('[DIL] Intercepted:', intercepted);
    if (intercepted) { 
      e.preventDefault(); 
      e.stopPropagation(); 
    }
  } catch (err) {
    console.error('[DIL] Error in handleClickEvent:', err);
  }
}

function handleAuxClickEvent(e) { 
  if (e.button === 1) handleClickEvent(e); 
}

function injectPageHook() { 
  const script = document.createElement('script'); 
  script.textContent = `(() => { 
    const FLAG='__dil_hook_installed__'; 
    if (window[FLAG]) return; 
    window[FLAG]=true; 
    const originalOpen=window.open; 
    const originalAssign=window.location.assign.bind(window.location); 
    const originalReplace=window.location.replace.bind(window.location); 
    const originalHrefDesc=Object.getOwnPropertyDescriptor(Location.prototype,'href'); 
    function intercept(url,opener){ 
      window.postMessage({ source:'dil-page', type:'REQUEST_NAV', url, opener }, '*'); 
    } 
    window.open=function(url){ 
      try{ 
        if(typeof url==='string'){ 
          intercept(url,'new-tab'); 
          return null; 
        } 
      }catch{} 
      return originalOpen.apply(window,arguments); 
    }; 
    Location.prototype.assign=function(url){ 
      try{ 
        intercept(String(url),'same-tab'); 
      }catch{} 
    }; 
    Location.prototype.replace=function(url){ 
      try{ 
        intercept(String(url),'same-tab'); 
      }catch{} 
    }; 
    if(originalHrefDesc && originalHrefDesc.set){ 
      Object.defineProperty(Location.prototype,'href',{ 
        get: originalHrefDesc.get, 
        set: function(v){ 
          try{ 
            intercept(String(v),'same-tab'); 
          }catch{} 
        }, 
        configurable:true, 
        enumerable: originalHrefDesc.enumerable 
      }); 
    } 
    window.addEventListener('message',(evt)=>{ 
      const d=evt&&evt.data; 
      if(!d||d.source!=='dil-content') return; 
      if(d.type==='PROCEED_NAV'){ 
        try{ 
          if(d.opener==='new-tab'){ 
            originalOpen(d.url,'_blank'); 
          } else { 
            originalAssign(d.url); 
          } 
        }catch(e){ 
          try{ 
            originalAssign(d.url); 
          }catch{} 
        } 
      } 
    }); 
  })();`; 
  (document.head || document.documentElement).appendChild(script); 
  script.remove(); 
}

window.addEventListener('message', (evt) => { 
  const d = evt && evt.data; 
  if (!d || d.source !== 'dil-page') return; 
  if (d.type === 'REQUEST_NAV') { 
    // 移除锚点检查，确保100%拦截
    // try { 
    //   const target = new URL(d.url, window.location.href); 
    //   const curr = new URL(window.location.href); 
    //   const anchorOnly = target.origin === curr.origin && target.pathname === curr.pathname && target.search === curr.search && target.hash !== curr.hash; 
    //   if (anchorOnly) return; 
    // } catch {} 
    handleIntercept(d.url, d.opener || 'same-tab'); 
  } 
});

function installInterceptors() { 
  document.addEventListener('click', handleClickEvent, { capture: true, passive: false }); 
  document.addEventListener('auxclick', handleAuxClickEvent, { capture: true, passive: false }); 
}

// 添加图片显示功能
function addFloatingImages() {
  if (!document.body) {
    // 如果body还没创建，等待一下
    setTimeout(addFloatingImages, 100);
    return;
  }
  
  // 检查是否已经添加过图片
  if (document.getElementById('__dil_left_img__') || document.getElementById('__dil_right_img__')) {
    return;
  }
  
  // 创建左图
  const leftImg = document.createElement('img');
  leftImg.id = '__dil_left_img__';
  leftImg.src = chrome.runtime.getURL('image/left.png');
  leftImg.style.position = 'fixed';
  leftImg.style.top = '20%';
  leftImg.style.left = '10px';
  leftImg.style.width = 'auto';
  leftImg.style.height = 'auto';
  leftImg.style.transform = 'scale(0.2)';
  leftImg.style.transformOrigin = 'left top';
  leftImg.style.zIndex = '2147483647';
  leftImg.style.pointerEvents = 'none';
  leftImg.style.maxWidth = 'none';
  leftImg.style.maxHeight = 'none';
  
  // 创建右图
  const rightImg = document.createElement('img');
  rightImg.id = '__dil_right_img__';
  rightImg.src = chrome.runtime.getURL('image/right.png');
  rightImg.style.position = 'fixed';
  rightImg.style.top = '20%';
  rightImg.style.right = '10px';
  rightImg.style.width = 'auto';
  rightImg.style.height = 'auto';
  rightImg.style.transform = 'scale(0.3)';
  rightImg.style.transformOrigin = 'right top';
  rightImg.style.zIndex = '2147483647';
  rightImg.style.pointerEvents = 'none';
  rightImg.style.maxWidth = 'none';
  rightImg.style.maxHeight = 'none';
  
  // 添加到页面
  document.body.appendChild(leftImg);
  document.body.appendChild(rightImg);
}

// 等待DOM加载完成后添加图片
function initFloatingImages() {
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', addFloatingImages);
  } else {
    addFloatingImages();
  }
  
  // 使用MutationObserver监听body创建
  if (!document.body) {
    const observer = new MutationObserver((mutations, obs) => {
      if (document.body) {
        addFloatingImages();
        obs.disconnect();
      }
    });
    observer.observe(document.documentElement, { childList: true, subtree: true });
  }
}

initFloatingImages();

injectPageHook(); 
installInterceptors();